import 'dart:async';
import 'dart:io';
import 'package:flutter/foundation.dart';
import 'package:flutter/material.dart';
import 'package:flutter_svg/svg.dart';
import 'package:http/http.dart' as http;
import 'package:just_audio/just_audio.dart';
import 'package:path_provider/path_provider.dart';
import 'package:permission_handler/permission_handler.dart';
import 'package:record/record.dart';
import 'package:shared_preferences/shared_preferences.dart';
import 'package:waveform_flutter/waveform_flutter.dart';

import '../../../Networking/ApiService.dart';
import '../../../Networking/DialogHelper.dart';
import '../../../Networking/getAuthHeaders.dart';
import '../DoctorDashboardScreen.dart';
import 'package:path/path.dart' as p;

class SoundDemo extends StatefulWidget {
  final Function(String path)? onStop; // Optional callback

  const SoundDemo({super.key, this.onStop});

  @override
  _SoundDemoState createState() => _SoundDemoState();
}

class _SoundDemoState extends State<SoundDemo> {
  final ApiService apiService = ApiService();
  late SharedPreferences sharedPreferences;
  final AudioRecorder _audioRecorder = AudioRecorder();
  final AudioPlayer _audioPlayer = AudioPlayer();
  final TextEditingController _audioNameController = TextEditingController();

  String? _filePath;
  bool _isRecording = false;
  bool _isRecorded = false;
  bool _isPlaying = false;
  bool _isPaused = false;
  bool _showUploadUI = false;

  int _recordDuration = 0;
  int _playbackDuration = 0;
  Duration _totalPlaybackDuration = Duration.zero;

  Timer? _timer;

  Waveform? _waveform;
  String Firstname="",USERID="",ROLEID="";

  @override
  void initState() {
    super.initState();

    _requestPermission();
    initialeclass();

    _audioPlayer.playerStateStream.listen((state) async {
      print('Player State: ${state.processingState}, playing: ${state.playing}');

      if (state.processingState == ProcessingState.completed) {
        _stopTimer();
        setState(() {
          _isPlaying = false;
          _isPaused = false;
          _playbackDuration = 0;
        });
      } else if (state.playing) {
        _startPlaybackTimer();
        setState(() {
          _isPlaying = true;
          _isPaused = false;
        });
      } else if (!state.playing) {
        _stopTimer();
        setState(() {
          _isPlaying = false;
          _isPaused = state.processingState == ProcessingState.ready;
        });
      }
    });

  }
  initialeclass() async {
    sharedPreferences = await SharedPreferences.getInstance();
    Firstname = sharedPreferences.getString("username") ?? '';
    USERID = sharedPreferences.getString("user_id") ?? '';
    ROLEID = sharedPreferences.getString("userRole") ?? '';

  }



  Future<void> _requestPermission() async {
    await Permission.microphone.request();
    await Permission.storage.request();
  }
  @override
  void dispose() {
    _audioRecorder.dispose();
    _audioPlayer.dispose();
    _timer?.cancel();
    super.dispose();
  }

  String _formatTime(int seconds) {
    final m = (seconds ~/ 60).toString().padLeft(2, '0');
    final s = (seconds % 60).toString().padLeft(2, '0');
    return '$m:$s';
  }

  void _startRecordingTimer() {
    _timer?.cancel();
    _timer = Timer.periodic(Duration(seconds: 1), (_) {
      setState(() => _recordDuration++);
    });
  }

  void _startPlaybackTimer() {
    _timer?.cancel();
    _timer = Timer.periodic(Duration(seconds: 1), (_) async {
      final position = await _audioPlayer.position;
      setState(() => _playbackDuration = position.inSeconds);
    });
  }

  void _stopTimer() {
    _timer?.cancel();
  }

  Future<void> _startRecording() async {
    if (await _audioRecorder.hasPermission()) {
      final dir = await getTemporaryDirectory();
      final path = '${dir.path}/audio_${DateTime.now().millisecondsSinceEpoch}.wav';

      await _audioRecorder.start(
        const RecordConfig(encoder: AudioEncoder.wav), // ✅ TRUE .wav
        path: path,
      );


      setState(() {
        _filePath = path;
        _isRecording = true;
        _recordDuration = 0;
        _showUploadUI = false;
        _isRecorded = false;
        _waveform = null;
      });

      _startRecordingTimer();
    }
  }

  Future<void> _stopRecording() async {
    await _audioRecorder.stop();
    _stopTimer();

    if (_filePath != null) {
      setState(() {
        _isRecording = false;
        _isRecorded = true;
        _showUploadUI = true;
      });
    }
  }

  Future<void> _playRecording() async {
    if (_filePath != null) {
      await _audioPlayer.setFilePath(_filePath!);
      _totalPlaybackDuration = await _audioPlayer.duration ?? Duration.zero;
      await _audioPlayer.play();
      setState(() {
        _playbackDuration = 0;
      });
    }
  }

  Future<void> _pauseRecording() async {
    await _audioPlayer.pause();
  }

  Future<void> _resumeRecording() async {
    await _audioPlayer.play();
  }

  Future<void> _uploadAudio() async {

    String name = _audioNameController.text.trim();

    if (_filePath == null || !_isRecorded) {
      ScaffoldMessenger.of(context)
          .showSnackBar(SnackBar(content: Text('No recorded file found')));
      return;
    }

    if (name.isEmpty) {
      ScaffoldMessenger.of(context)
          .showSnackBar(SnackBar(content: Text('Please enter audio name')));
      return;
    }

    try {

      DialogHelper.showLoading(context);
      // Replace spaces with underscores
      String cleanedFileName = name.replaceAll(' ', '_') + p.extension(_filePath!);
      print("=== Uploading Audio ===");
      print("Audio Name: $name");
      print("Cleaned File Name: $cleanedFileName");
      print("File Path: $_filePath");
      print("User ID: $USERID");
      print("API Endpoint: ${apiService.BaseURL + apiService.VOICEUPLOAD}");

      final headers = await getAuthHeaders();
      var request = http.MultipartRequest(
        'POST',
        Uri.parse(apiService.BaseURL + apiService.VOICEUPLOAD ), // Replace with your endpoint
      );
      request.headers.addAll(headers);

      request.files.add(
        await http.MultipartFile.fromPath(
          'medical_audio',
          _filePath!,
          filename: cleanedFileName,
        ),
      );
      request.fields['user_id'] = USERID;
      request.fields['file_name'] = name;// Replace with actual user ID

      var response = await request.send();

      if (response.statusCode == 200) {
        DialogHelper.dismissDialog();
        bool? uploaded = await showDialog(
          context: context,
          barrierDismissible: false,
          builder: (BuildContext context) {
            Future.delayed(Duration(seconds: 2), () {
              if (Navigator.canPop(context)) {
                Navigator.pop(context, true); // Return true to calling code
              }
            });

            return WillPopScope(
              onWillPop: () async => false,
              child: AlertDialog(
                insetPadding: EdgeInsets.zero,
                contentPadding: EdgeInsets.zero,
                shape: RoundedRectangleBorder(
                  borderRadius: BorderRadius.circular(16),
                ),
                content: Container(
                  decoration: BoxDecoration(
                    color: Colors.green.shade50,
                    borderRadius: BorderRadius.circular(16),
                  ),
                  padding: EdgeInsets.all(20),
                  child: Column(
                    mainAxisSize: MainAxisSize.min,
                    children: [
                      Icon(Icons.check_circle, color: Colors.green, size: 50),
                      SizedBox(height: 12),
                      Text(
                        'Audio Uploaded Successfully',
                        style: TextStyle(
                            fontSize: 18,
                            fontWeight: FontWeight.bold,
                            color: Colors.green),
                        textAlign: TextAlign.center,
                      ),
                    ],
                  ),
                ),
              ),
            );
          },
        );

        if (uploaded == true) {
          setState(() {
            _audioNameController.clear();
            _filePath = null;
            _filePath = null;
            _isRecorded = false;
            _recordDuration = 0;
            _playbackDuration = 0;
            _showUploadUI = false;
            _waveform = null;

          });
        }
      } else {
        DialogHelper.dismissDialog();
        ScaffoldMessenger.of(context).showSnackBar(
          SnackBar(content: Text('Upload failed (Status: ${response.statusCode})')),
        );
      }
    } catch (e) {
      DialogHelper.dismissDialog();
      ScaffoldMessenger.of(context).showSnackBar(
        SnackBar(content: Text('Upload error: $e')),
      );
    }
  }


  Widget _actionButton(IconData icon, Color color, VoidCallback? onTap) {
    return IconButton(
      icon: Icon(icon, color: color),
      iconSize: 40,
      onPressed: onTap,
    );
  }
  Future<void> _stopPlayback() async {
    await _audioPlayer.stop();
    _stopTimer();
    setState(() {
      _isPlaying = false;
      _playbackDuration = 0;
    });
  }
  @override
  Widget build(BuildContext context) {
    return       PopScope(
      canPop: false, // prevent default back
      onPopInvoked: (didPop) async {
        if (!didPop) {
          // Replace current screen with another one
          Navigator.pushReplacement(
            context,
            MaterialPageRoute(builder: (context) => DoctorDashboardScreen()),
          );
        }
      },
      child: Scaffold(
        backgroundColor: Color(0xfff5f5f5),
        appBar: AppBar(
          backgroundColor: Color(0xff1F5C93),
          leading: IconButton(
            icon: Icon(Icons.arrow_back_ios_new_outlined, color: Colors.white),
            onPressed: () {
              Navigator.pop(context);
            },
          ),
          title: Text('Record', style: TextStyle(color: Colors.white)),
        ),
        body: SingleChildScrollView(
          child: Column(
            children: [
              Container(
                margin: EdgeInsets.all(16),
                padding: EdgeInsets.symmetric(vertical: 40),
                decoration: BoxDecoration(
                  color: Colors.white,
                  borderRadius: BorderRadius.circular(32),
                ),
                width: double.infinity,
                child: Column(
                  children: [
                    Text(
                      _isPlaying || _isPaused
                          ? _formatTime(_playbackDuration)
                          : _formatTime(_recordDuration),
                      style: TextStyle(fontSize: 36, fontWeight: FontWeight.bold),
                    ),
                    SizedBox(height: 10),
                    if (_isRecording || _isPlaying)
                    Padding(
                      padding: const EdgeInsets.all(8.0),
                      child: LinearProgressIndicator(
                        value: _isRecording
                            ? (_recordDuration / 600.0).clamp(0.0, 1.0) // Assuming 60s max for visual
                            : _isPlaying || _isPaused
                            ? (_totalPlaybackDuration.inSeconds > 0
                            ? _playbackDuration / _totalPlaybackDuration.inSeconds
                            : 0.0)
                            : null,
                        minHeight: 8,
                        backgroundColor: Colors.grey[300],
                        color: _isRecording ? Colors.red : Colors.blue,
                      ),
                    ),

                  ],
                ),
              ),
              Padding(
                padding: const EdgeInsets.all(16.0),
                child: Container(
                  decoration: BoxDecoration(
                    color: Colors.white,
                    borderRadius: BorderRadius.circular(20),
                    boxShadow: [
                      BoxShadow(color: Colors.black12, blurRadius: 10, offset: Offset(0, 4)),
                    ],
                  ),
                  padding: const EdgeInsets.symmetric(vertical: 20, horizontal: 16),
                  child: Row(
                    mainAxisAlignment: MainAxisAlignment.spaceAround,
                    children: [
                      if (!_isRecording && !_isPlaying)
                        _actionButton(Icons.mic, Colors.blue, _startRecording),

                      if (_isRecorded && !_isPlaying)
                        _actionButton(Icons.delete_outline, Colors.red, () {
                          if (_filePath != null) {
                            File(_filePath!).deleteSync();
                            setState(() {
                              _filePath = null;
                              _isRecorded = false;
                              _recordDuration = 0;
                              _playbackDuration = 0;
                              _showUploadUI = false;
                              _waveform = null;
                            });
                          }
                        }),

                      if (_isRecording)
                        _actionButton(Icons.stop, Colors.red, _stopRecording),

                      if (_isRecorded && !_isPlaying)
                        _actionButton(Icons.play_arrow, Colors.green, _playRecording),

                      if (_isPlaying)
                        _actionButton(Icons.stop, Colors.red, _stopPlayback), // New stop playback button
                    ],
                  ),

                ),
              ),
              if (_showUploadUI)
                Padding(
                  padding: const EdgeInsets.all(16.0),
                  child: Column(
                    children: [
                      TextField(
                        controller: _audioNameController,
                        decoration: InputDecoration(
                          labelText: 'Audio Name',
                          border: OutlineInputBorder(),
                        ),
                      ),
                      SizedBox(height: 10),
                      ElevatedButton.icon(
                        onPressed: _isPlaying ? null : _uploadAudio,
                        icon: Icon(Icons.cloud_upload,color: Colors.white,),
                        label: Text('Upload Audio',style: TextStyle(color: Colors.white),),
                        style: ElevatedButton.styleFrom(
                          backgroundColor: Colors.green,
                          shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
                        ),
                      ),
                    ],
                  ),
                ),
            ],
          ),
        ),
      ),
    );
  }
}